/**
 * Parallel FindAll API integration service
 * Handles event discovery and enrichment for Middle East events
 */

import { FindAllCandidate, FindAllResult, FindAllRunStatus } from '@/types/event';

const PARALLEL_API_BASE = 'https://api.parallel.ai/v1beta';
const API_KEY = process.env.PARALLEL_API_KEY || '';

/**
 * Headers required for all Parallel API requests
 */
function getHeaders(): HeadersInit {
  return {
    'x-api-key': API_KEY,
    'parallel-beta': 'findall-2025-09-15',
    'Content-Type': 'application/json',
  };
}

/**
 * Ingest a natural language query to get structured schema
 */
export async function ingestQuery(objective: string): Promise<{
  objective: string;
  entity_type: string;
  match_conditions: Array<{ name: string; description: string }>;
}> {
  const response = await fetch(`${PARALLEL_API_BASE}/findall/ingest`, {
    method: 'POST',
    headers: getHeaders(),
    body: JSON.stringify({ objective }),
  });

  if (!response.ok) {
    throw new Error(`Ingest failed: ${response.statusText}`);
  }

  return response.json();
}

/**
 * Create a FindAll run to discover Middle East events
 */
export async function createFindAllRun(options: {
  objective: string;
  entityType?: string;
  matchConditions?: Array<{ name: string; description: string }>;
  enrichments?: Array<{ name: string; description: string }>;
  generator?: 'base' | 'core' | 'pro';
  matchLimit?: number;
}): Promise<{ findall_id: string }> {
  const {
    objective,
    entityType = 'events',
    matchConditions,
    enrichments,
    generator = 'core',
    matchLimit = 50,
  } = options;

  const body: Record<string, unknown> = {
    objective,
    entity_type: entityType,
    generator,
    match_limit: matchLimit,
  };

  if (matchConditions) {
    body.match_conditions = matchConditions;
  }

  if (enrichments) {
    body.enrichments = enrichments;
  }

  const response = await fetch(`${PARALLEL_API_BASE}/findall/runs`, {
    method: 'POST',
    headers: getHeaders(),
    body: JSON.stringify(body),
  });

  if (!response.ok) {
    const error = await response.text();
    throw new Error(`CreateRun failed: ${response.statusText} - ${error}`);
  }

  return response.json();
}

/**
 * Check the status of a FindAll run
 */
export async function getRunStatus(findallId: string): Promise<FindAllRunStatus> {
  const response = await fetch(`${PARALLEL_API_BASE}/findall/runs/${findallId}`, {
    method: 'GET',
    headers: getHeaders(),
  });

  if (!response.ok) {
    throw new Error(`GetStatus failed: ${response.statusText}`);
  }

  return response.json();
}

/**
 * Get the results of a completed FindAll run
 */
export async function getRunResult(findallId: string): Promise<FindAllResult> {
  const response = await fetch(`${PARALLEL_API_BASE}/findall/runs/${findallId}/result`, {
    method: 'GET',
    headers: getHeaders(),
  });

  if (!response.ok) {
    throw new Error(`GetResult failed: ${response.statusText}`);
  }

  return response.json();
}

/**
 * Poll for run completion with timeout
 */
export async function pollUntilComplete(
  findallId: string,
  maxWaitMs: number = 300000, // 5 minutes default
  pollIntervalMs: number = 5000 // 5 seconds
): Promise<FindAllRunStatus> {
  const startTime = Date.now();

  while (Date.now() - startTime < maxWaitMs) {
    const status = await getRunStatus(findallId);

    if (!status.status.is_active) {
      return status;
    }

    // Wait before next poll
    await new Promise(resolve => setTimeout(resolve, pollIntervalMs));
  }

  throw new Error('Polling timeout exceeded');
}

/**
 * Run a complete FindAll search for Middle East events
 * This is the main entry point for event discovery
 */
export async function findMiddleEastEvents(): Promise<FindAllCandidate[]> {
  // Get the current date and calculate date range for past month
  const today = new Date();
  const pastMonth = new Date(today);
  pastMonth.setMonth(pastMonth.getMonth() - 1);
  
  const futureMonth = new Date(today);
  futureMonth.setMonth(futureMonth.getMonth() + 3);

  const objective = `FindAll major B2B and B2C events happening in the Middle East region (UAE, Saudi Arabia, Qatar, Bahrain, Kuwait, Oman, Jordan, Egypt, Israel, Lebanon) from ${pastMonth.toISOString().split('T')[0]} to ${futureMonth.toISOString().split('T')[0]} with at least 1000 expected attendees. Include conferences, exhibitions, summits, festivals, trade shows, concerts, and sports events.`;

  const matchConditions = [
    {
      name: 'middle_east_location',
      description: 'Event must be held in a Middle East country (UAE, Saudi Arabia, Qatar, Bahrain, Kuwait, Oman, Jordan, Egypt, Israel, Lebanon)'
    },
    {
      name: 'minimum_attendees',
      description: 'Event must have at least 1000 expected or confirmed attendees'
    },
    {
      name: 'valid_event_type',
      description: 'Event must be a B2B or B2C event such as conference, exhibition, summit, festival, trade show, concert, sports event, or networking event'
    }
  ];

  const enrichments = [
    { name: 'event_name', description: 'Official name of the event suitable for display' },
    { name: 'event_date', description: 'Start date of the event in YYYY-MM-DD format' },
    { name: 'event_end_date', description: 'End date of the event in YYYY-MM-DD format if multi-day' },
    { name: 'attendees_range', description: 'Expected number of attendees as a range (e.g., "5000-10000")' },
    { name: 'venue_name', description: 'Name of the venue where the event is held' },
    { name: 'city', description: 'City where the event is held' },
    { name: 'country', description: 'Country where the event is held' },
    { name: 'ticket_available', description: 'Whether tickets are currently available (yes/no)' },
    { name: 'ticket_pricing', description: 'Ticket pricing information if available' },
    { name: 'one_liner_description', description: 'One sentence description of the event' },
    { name: 'attendee_profiles', description: 'Description of typical attendees (e.g., "Tech executives, startups, investors")' },
    { name: 'sponsors', description: 'List of major sponsors if any, comma separated' },
    { name: 'event_type', description: 'Type of event: conference, exhibition, summit, festival, trade-show, concert, sports, networking, workshop, or other' },
    { name: 'website_url', description: 'Official website URL of the event' }
  ];

  // Create the FindAll run
  const { findall_id } = await createFindAllRun({
    objective,
    entityType: 'events',
    matchConditions,
    enrichments,
    generator: 'pro',
    matchLimit: 50,
  });

  console.log(`FindAll run created: ${findall_id}`);

  // Poll until complete
  const finalStatus = await pollUntilComplete(findall_id);
  console.log(`FindAll run completed with status: ${finalStatus.status.status}`);

  // Get results
  const result = await getRunResult(findall_id);

  // Filter for matched candidates only
  return result.candidates.filter(c => c.match_status === 'matched');
}

/**
 * Transform a FindAll candidate into our Event format
 */
export function transformCandidateToEvent(candidate: FindAllCandidate, index: number): import('@/types/event').Event {
  const output = candidate.output || {};

  const getValue = (key: string, defaultValue: string = ''): string => {
    return output[key]?.value || defaultValue;
  };

  const eventType = getValue('event_type', 'other').toLowerCase().replace(/\s+/g, '-') as import('@/types/event').EventType;
  const validTypes = ['conference', 'exhibition', 'summit', 'festival', 'trade-show', 'networking', 'workshop', 'concert', 'sports', 'other'];

  return {
    id: candidate.candidate_id || `event-${index}`,
    name: getValue('event_name', candidate.name),
    date: getValue('event_date'),
    endDate: getValue('event_end_date') || undefined,
    attendeesRange: getValue('attendees_range', '1000+'),
    venue: getValue('venue_name', 'TBA'),
    city: getValue('city', ''),
    country: getValue('country', 'Middle East'),
    ticketAvailable: getValue('ticket_available', 'unknown').toLowerCase() === 'yes',
    pricing: getValue('ticket_pricing') || undefined,
    description: getValue('one_liner_description', candidate.description || ''),
    attendeeProfiles: getValue('attendee_profiles', 'Business professionals and industry experts'),
    sponsors: getValue('sponsors', '').split(',').map(s => s.trim()).filter(Boolean),
    eventType: validTypes.includes(eventType) ? eventType : 'other',
    websiteUrl: getValue('website_url', candidate.url) || undefined,
    createdAt: new Date().toISOString(),
    updatedAt: new Date().toISOString(),
  };
}

